import asyncio
import importlib
import os
import fnmatch
import time as t
import socket
from datetime import datetime as dt
from py_pli.pylib import EndPointType
from py_pli.pylib import URPCFunctions
from py_pli.pylib import VUnits
from py_pli.pylib import send_msg
from urpc_enum.corexymoverparameter import CoreXYMoverParameter
from urpc_enum.measurementparameter import MeasurementParameter
from urpc_enum.moverparameter import MoverParameter
from urpc_enum.nodeparameter import NodeParameter
from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
import config_enum.scan_table_enum as scan_table_enum
import fleming.mleuschner.hw_test1 as hw
#import userscripts.mleuschner.hw_test1 as hw

# 2021-29-11/Mirko Leuschner
# PMT Test

#maximum high voltage for PMT
hvmax=188

as1pos=-1
as2pos=-1
bldpos=-1
elspos=-1
fmpos=-1
fmspos=-1

#Bremen values 
bldpd = 200.66 #351.46  #220124
bldtop = 90
bldbottom = 35.25

# Bremen Values 
elsp1 = 23.8 #23.6 #24.07    #220124
elsp2 = 203.8 #203.6 #204.08   #220124
els_al = 243.6 #242.4 #249 #111.375 #245.85

fmsdiag  = 35.80
fms0     = 31.422   
fmsoffset = 24.075

# Report Functions

def instrument():
    return socket.gethostname()

def reportpath():
    mode = 0o777
    path, filename = os.path.split(__file__)
    md1=dt.now().strftime('%y%m%d')
    fpath = os.path.join(path, instrument()+md1)   #Report File Path
    if not os.path.isdir(fpath):
        os.mkdir(fpath, mode)
    os.chmod(fpath, mode)
    return(os.path.join(fpath, 'Report'+md1+'.txt'))

def datapath(fl='Test'):
    path, filename = os.path.split(reportpath())
    md1=dt.now().strftime('%y%m%d')
    fpath = path+'/'   #Output File Path
    fl =fl+md1+'_Run'
    no = len(fnmatch.filter(os.listdir(fpath), fl+'*.txt'))+1
    fname1 = str.format(fl+'{0:03d}.txt', no)
    return(fpath+fname1)

def report(text='', source='Note'):
    mode = 0o777
    rfp = reportpath()
    if os.path.exists(rfp):
        os.chmod(rfp, mode)
    mt1=dt.now().strftime('%H:%M:%S')
    md1=dt.now().strftime('%y%m%d')
    with open(rfp, 'a') as report:
        report.write('\n'+mt1+'; '+md1+'; '+source+'; '+text+'\n')
    os.chmod(rfp, mode)

# Init Functions

async def init():
    print("Startup Hardware \n")
    await VUnits.instance.hal.StartupHardware()
    print("Hardware started \n")
    await VUnits.instance.hal.InitializeDevice()
    print("Device initialized.\n")
    await VUnits.instance.hal.HomeMovers()
    print("Movers homed.\n")
    await VUnits.instance.hal.TurnLedsOff()
    print("LEDs off.\n")
    await pd_close()
    print("PlateDoor closed.\n")


async def ws_init():
    await al_off()
    await fms_diag()
    await els_2()
    await bld_pd()
    await st_measpos()

async def go():
    print("Instrument is {}".format(instrument()))
    report('Init', 'pmt_go')
    await init()
    await hw.start_al()
    await ws_init()


async def fisetup(fifi=3, fifo=8.2):
    '''
    rework 211129
    '''
    myname = '.pmt.fisetup'
    await st_measpos()
    #await st_platetype("96_Opti_Plate")
    await st_platetype("'384 OptiPlate (Black)'")
    await fms_set(fifi)
    await fm(fifo)
    result = str.format("Filter Pos.: {0}; Focus Pos.: {1}", fifi, fifo) 
    report(result, myname)
    return (0)

# Mover Functions

async def as1_home():
    as1 = VUnits.instance.hal.detectorApertureSlider1
    return await as1.Home()

async def as1(pos):
    # Move ApertureSlider1 to absolute position
    as1 = VUnits.instance.hal.detectorApertureSlider1
    await as1.UseProfile(1)
    await as1.Move(pos)
    return await as1.GetPosition()

async def as1_1():
    # Move ApertureSlider1 to 1.6mm Aperture-Position
    as1 = VUnits.instance.hal.detectorApertureSlider1
    await as1.UseProfile(1)
    await as1.MoveTo16Pos()
    return await as1.GetPosition()

async def as1_3():
    # Move ApertureSlider1 to 3.0mm Aperture-Position
    as1 = VUnits.instance.hal.detectorApertureSlider1
    await as1.UseProfile(1)
    await as1.MoveTo30Pos()
    return await as1.GetPosition()

async def as1_getpos():
    # get ApertureSlider1-Position
    as1 = VUnits.instance.hal.detectorApertureSlider1
    return await as1.GetPosition()

async def as1_close():
    # close position
    as1 = VUnits.instance.hal.detectorApertureSlider1
    await as1.Home()
    return await as1.GetPosition()

async def as2_home():
    as2 = VUnits.instance.hal.detectorApertureSlider2
    return await as2.Home()

async def as2(pos):
    # Move ApertureSlider2 to absolute position
    as2 = VUnits.instance.hal.detectorApertureSlider2
    await as2.UseProfile(1)
    await as2.Move(pos)
    return await as2.GetPosition()

async def as2_1():
    # Move ApertureSlider2 to 1.6mm Aperture-Position
    as2 = VUnits.instance.hal.detectorApertureSlider2
    await as2.UseProfile(1)
    await as2.MoveTo16Pos()
    return await as2.GetPosition()

async def as2_3():
    # Move ApertureSlider2 to 3.0mm Aperture-Position
    as2 = VUnits.instance.hal.detectorApertureSlider2
    await as2.UseProfile(1)
    await as2.MoveTo30Pos()
    return await as2.GetPosition()

async def as2_getpos():
    # get ApertureSlider2-Position
    as2 = VUnits.instance.hal.detectorApertureSlider2
    return await as2.GetPosition()

async def as2_close():
    # close position
    as2 = VUnits.instance.hal.detectorApertureSlider2
    await as2.Home()
    return await as2.GetPosition()

async def asx_home():
    return await as1_home(), await as2_home()

async def asx(pos):
    return await as1(pos), await as2(pos)

async def asx_getpos():
    return await as1_getpos(), await as2_getpos()

async def asx_1():
    return await as1_1(), await as2_1()

async def asx_close():
    return await as1_close(), await as2_close()

async def asx_3():
    return await as1_3(), await as2_3()

async def bld_home():
    bld = VUnits.instance.hal.bottomLightDirector
    return await bld.Home()

async def bld(pos):
    # move BLD to position
    bld = VUnits.instance.hal.bottomLightDirector
    await bld.UseProfile(1)
    await bld.Move(pos)
    return await bld.GetPosition()

async def bld_getpos():
    # get BLD-Position
    bld = VUnits.instance.hal.bottomLightDirector
    return await bld.GetPosition()

async def bld_pd():
    return await bld(bldpd)

async def bld_bottom():
    return await bld(bldbottom)

async def bld_top():
    return await bld(bldtop)

async def els_home():
    els = VUnits.instance.hal.excitationLightSelector
    return await els.Home()    

async def els(pos):
    # For adjustment
    els = VUnits.instance.hal.excitationLightSelector
    await els.UseProfile(1)
    await els.Move(pos)
    return await els.GetPosition()

async def els_getpos():
    # get ELS-Position
    els = VUnits.instance.hal.excitationLightSelector
    return await els.GetPosition()

async def els_1():
    # Move ELS to increase beam diameter
    return await els(elsp1)

async def els_2():
    # Move ELS to decrease beam diameter
    return await els(elsp2)

async def els_a():
    # Move ELS to Alpha position
    return await els(els_al)

async def fm_home():
    fm = VUnits.instance.hal.focusMover
    return await fm.Home()

async def fm(pos):
    fm = VUnits.instance.hal.focusMover
    await fm.UseProfile(1)
    await fm.Move(pos)
    return await fm.GetPosition()

async def fm_getpos():
    # get FM-Position
    fm = VUnits.instance.hal.focusMover
    return await fm.GetPosition()

async def fms_home():
    fms = VUnits.instance.hal.filterModuleSlider
    return await fms.Home()

async def fms(pos):
    fms = VUnits.instance.hal.filterModuleSlider
    await fms.UseProfile(1)
    await fms.Move(pos)
    return await fms.GetPosition()

async def fms_getpos():
    # get FMS-Position
    fms = VUnits.instance.hal.filterModuleSlider
    return await fms.GetPosition()

async def fms_diag():
    return await fms(fmsdiag)

async def fms_set(filter_nr):
    position = fms0 + filter_nr * fmsoffset
    return await fms(position)

async def pd_home():
    return await VUnits.instance.hal.plateDoor.Home()
    
async def pd_open():
    pd = VUnits.instance.hal.plateDoor
    await pd.UseProfile(1)    
    await pd.Open()    
    print("Plate Door open.")

async def pd_close():
    pd = VUnits.instance.hal.plateDoor
    await pd.UseProfile(1)    
    await pd.Close()    
    print("Plate Door closed.")   

async def st_home():
    st = VUnits.instance.hal.scan_table
    await st.Home()
    print(f"Scan Table X Step Errors: {st.CoreXY.LastHomeStepErrors[0]}")
    print(f"Scan Table Y Step Errors: {st.CoreXY.LastHomeStepErrors[1]}")
    return st.CoreXY.LastHomeStepErrors

async def st_getposition():
    st = VUnits.instance.hal.scan_table
    return await st.GetPosition()

async def st_move(x=0, y=0):
    st = VUnits.instance.hal.scan_table
    await st.Move(x, y)
    return await st.GetPosition()
     
async def eject():
    await asx_close()
    await st_move(10, 180) #PreLoadPos
    await pd_open()
    await st_move(10, 300) #RightLoadPos
    report('Plate removed','eject')
    print(f"Remove Plate")
     
async def load():
    await st_move(10, 180) #PreLoadPos
    await pd_close()
    await st_home()
    report('Plate loaded','load')
    print(f"Plate loaded")
     
async def st_movetowell(x=1, y=1, xo=0, yo=0):
    st = VUnits.instance.hal.scan_table
    await st.MoveToWell(x, y, xo, yo)
    return await st.GetPosition()
     
async def st_platetype(pt="1536_EnSpire_Adjusting_Plate"):
    st = VUnits.instance.hal.scan_table
    st.load_plates()
    st.SetPlateType(pt)
    report('PlateType: '+pt,'st_platetype')
    return pt

async def st_measpos(mp="FBD_Top"):
    st = VUnits.instance.hal.scan_table
    st.SetCurrentMeasPosition(mp)
    report('Meas Pos: '+mp,'st_measpos')
    return mp
   
# Alpha Functions

async def al_on():
    # Reset Signal to turn on!!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'alphaon'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.Alpha)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    #PyLogger.logger.info('=============Alpha Laser On!!!==================')
    print('=============Alpha-Laser On!!!==================')

async def al_off():
    #Set Signal to turn off!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'alphaoff'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.Alpha)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    #PyLogger.logger.info('=============Alpha Laser Off!!!==================')
    print('=============Alpha-Laser Off!!!==================')
    
# PMT Functions

async def aoreset():
    address = 0
    sequence = [
        0x0E000000 | ((4 & 7) << 15) | ((4 & 7) << 12) | ((4 & 7) << 3) | (4 & 7),
        0x00000000,
    ]
    meas = VUnits.instance.hal.measurementUnit.MeasurementFunctions
    await meas.WriteSequence(address, len(sequence), sequence, timeout=5)
    await meas.StartSequence(address, timeout=1)
    await asyncio.sleep(0.1)
    if (await meas.GetStatus(timeout=1))[0] & 0x01:
        return f"aoreset() passed"
    else:
        return f"aoreset() failed"
    return(0)

async def hvpmt1on():
    # Reset Signal to turn on!!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt1on'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVOnPMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    #PyLogger.logger.info('=============HV PMT1 On!!!==================')

async def hvpmt2on():
    # Reset Signal to turn on!!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt2on'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVOnPMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    #PyLogger.logger.info('=============HV PMT2 On!!!==================')

async def hvpmt1off():
    #Set Signal to turn off!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt1off'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVOnPMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    #PyLogger.logger.info('=============HV PMT1 Off!!!==================')

async def hvpmt2off():
    #Set Signal to turn off!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt2off'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVOnPMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    #PyLogger.logger.info('=============HV PMT2 Off!!!==================')

async def hvgate1on():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate1on'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVGatePMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

async def hvgate2on():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate2on'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

async def hvgate1off():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate1off'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

async def hvgate2off():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate2off'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

async def pmt1setdl(dl):
    # set dl-value
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 DL Set==================')

async def pmt1sethv(hv):
    # set hv-value
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, hv, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Set==================')

async def pmt1hvenable():
    # enable hv
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 1, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Enabled==================')

async def pmt1hvdisable():
    # enable hv
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 0, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Disabled==================')

async def pmt2setdl(dl):
    # set dl-value
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, dl, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 DL Set==================')

async def pmt2sethv(hv):
    # set hv-value
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, hv, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Set==================')

async def pmt2hvenable():
    # enable hv
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, 1, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Enabled==================')

async def pmt2hvdisable():
    # enable hv
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, 0, timeout=1)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============PMT1 HV Disabled==================')

async def flashsetpower(power, scale=4096):
    # set flash power
    delay = 0.1
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power/scale, timeout=1)
    #await hw.flash_setpw(power)
    await asyncio.sleep(delay)
    #PyLogger.logger.info('=============Flash Power Set==================')



# PMT Test-Functions

async def dlscan(dlstart=0, dlstop=255, dlstep=1, window=100, pre_cnt_window=100):
    '''
    ml 211025:
    discrscan, dual channel, no hv
    rework 211129
    '''
    myname = '.pmt.dlscan'

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    #pre_cnt_window = 100    # 1µs
    wscale = 100000 // pre_cnt_window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    delay = 0.1
    dacscale1 = 256
    await pmt1sethv(0)
    await pmt2sethv(0)
    await pmt1hvdisable()
    await pmt2hvdisable()
    await hvgate1on()
    await hvgate2on()

    op_id = 'discrscan'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 2)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    if dlstop>255: 
        dlstop=255
    if dlstart>(dlstop-dlstep):
        dlstart=dlstop-dlstep
    dfp = datapath('DLScan')
    path, filename = os.path.split(dfp)
    result = str.format("File: {0}; Start: {1}; Stop: {2}; Step: {3}; Window: {4}; PreCount: {5}",filename, dlstart, dlstop, dlstep, window, pre_cnt_window)
    report(result, myname)
    result = str.format("File: {0}; wScale: {1}; cntWin: {2}; winCorse: {3}; winFine: {4}",filename, wscale, cnt_window, window_corse, window_fine)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"dac\tcount1\tcount2\n")
        for i in range(dlstart, dlstop+dlstep, dlstep):
            await pmt1setdl( i/dacscale1)
            await pmt2setdl( i/dacscale1)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                print(f"{i}\t{results[0]}\t{results[1]}")
                data.write(f"{i}\t{results[0]}\t{results[1]}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    report('Done', myname)
    print('Done')
    return(0)

async def dlscan2(hv=100, dlstart=0, dlstop=255, dlstep=1, window=100, pre_cnt_window=100):
    '''
    ml 211025:
    discrscan, dual channel, hv on
    rework 211129
    '''
    myname = '.pmt.dlscan2'

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    #pre_cnt_window = 100    # 1µs
    wscale = 100000 // pre_cnt_window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    delay = 0.1
    dacscale1 = 256
    await pmt1sethv(hv/256)
    await pmt2sethv(hv/256)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    op_id = 'discrscan'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 2)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    if dlstop>255: 
        dlstop=255
    if dlstart>(dlstop-dlstep):
        dlstart=dlstop-dlstep
    dfp = datapath('DLScan')
    path, filename = os.path.split(dfp)
    result = str.format("File: {0}; Start: {1}; Stop: {2}; Step: {3}; Window: {4}; PreCount: {5}; HV: {6}",filename, dlstart, dlstop, dlstep, window, pre_cnt_window, hv)
    report(result, myname)
    result = str.format("File: {0}; wScale: {1}; cntWin: {2}; winCorse: {3}; winFine: {4}",filename, wscale, cnt_window, window_corse, window_fine)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"dac\tcount1\tcount2\n")
        for i in range(dlstart, dlstop+dlstep, dlstep):
            await pmt1setdl( i/dacscale1)
            await pmt2setdl( i/dacscale1)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                print(f"{i}\t{results[0]}\t{results[1]}")
                data.write(f"{i}\t{results[0]}\t{results[1]}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    report('Done', myname)
    print('Done')
    return(0)

async def dlscan3(dlstart=0, dlstop=255, dlstep=1, window=100, pre_cnt_window=100):
    '''
    ml 211210:
    discrscan, dual channel, no hv, with input gating
    '''
    myname = '.pmt.dlscan3'

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    #pre_cnt_window = 100    # 1µs
    wscale = 100000 // pre_cnt_window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    delay = 0.1
    dacscale1 = 256
    await pmt1sethv(0)
    await pmt2sethv(0)
    await pmt1hvdisable()
    await pmt2hvdisable()
    await hvgate1on()
    await hvgate2on()

    op_id = 'discrscan'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 2)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    if dlstop>255: 
        dlstop=255
    if dlstart>(dlstop-dlstep):
        dlstart=dlstop-dlstep
    dfp = datapath('DLScan')
    path, filename = os.path.split(dfp)
    result = str.format("File: {0}; Start: {1}; Stop: {2}; Step: {3}; Window: {4}; PreCount: {5}",filename, dlstart, dlstop, dlstep, window, pre_cnt_window)
    report(result, myname)
    result = str.format("File: {0}; wScale: {1}; cntWin: {2}; winCorse: {3}; winFine: {4}",filename, wscale, cnt_window, window_corse, window_fine)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"dac\tcount1\tcount2\n")
        for i in range(dlstart, dlstop+dlstep, dlstep):
            await pmt1setdl( i/dacscale1)
            await pmt2setdl( i/dacscale1)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                print(f"{i}\t{results[0]}\t{results[1]}")
                data.write(f"{i}\t{results[0]}\t{results[1]}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    report('Done', myname)
    print('Done')
    return(0)

async def hvscan(dl=64, hvstart=0, hvstop=188, hvstep=1, flashes=10, flash_mode=1, pre_cnt_window=10):
    '''
    ml 211027:
    hvscan, dual channel
    rework 211129
    with flash
    '''
    myname = '.pmt.hvscan'

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    dacscale1 = 256
    await pmt1setdl( dl/dacscale1)
    await pmt2setdl( dl/dacscale1)
    await pmt1sethv( 0)
    await pmt2sethv( 0)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()
    await pmt1sethv( hvstart/dacscale1)
    await pmt2sethv( hvstart/dacscale1)

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.3
    frequency = 100
    #flash_mode = 1
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest2a'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    if hvstop>hvmax: 
        hvstop=hvmax
    if hvstart>(hvstop-hvstep):
        hvstart=hvstop-hvstep
    dfp = datapath('HVScan')
    path, filename = os.path.split(dfp)
    result = str.format("File: {0}; DL: {1}; Start: {2}; Stop: {3}; Step: {4}; Flashes: {5}; Flash_Mode: {6}; PreCount: {7}; cntWin: {8}",filename, dl, hvstart, hvstop, hvstep, flashes, flash_mode, pre_cnt_window, cnt_window)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"dac\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(hvstart, hvstop+hvstep, hvstep):
            await pmt1sethv( i/dacscale1)
            await pmt2sethv( i/dacscale1)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    await pmt1sethv(0)
    await pmt2sethv(0)
    report('Done', myname)
    print('Done')
    return(0)

async def darkcount(iterations, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, window=1000, pre_cnt_window=10000):
    '''
    ml 210906: darkcount measurement
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    highresetdelay = 5000 # 50µs
    lowresetdelay = 5000  # 50µs
    meastime = 5000       # 50µs
    delaytime = 5000      # 50µs
    integdelay = 500      # 5µs
    #pre_cnt_window = 100    # 1µs
    wscale = 100000 // pre_cnt_window   # 1ms scale for window
    cnt_window = window * wscale
    window_corse, window_fine = divmod(cnt_window, 65536)

    op_id = 'darktest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.TimerWaitAndRestart(highresetdelay)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    # measurement
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)

    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


    result = str.format("iterations: {0}, Count-Window: {1}, PreCount-Window: {2}",iterations, cnt_window, pre_cnt_window)
    report(result, '.pmt.darkcount1')
    dfp = datapath('DarkCount')
    path, filename = os.path.split(dfp)
    report(filename, '.pmt.darkcount1'); #Write to Report
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"iter\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\tTime\n")
        for i in range(iterations):
            try:
                time_stamp1 = dt.now().strftime('%H:%M:%S.%f')                
                await measurement_unit.ExecuteMeasurement(op_id)
                await asyncio.sleep(delay)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                print(f"{i}\t{results[0]}\t{results[5]}")
                data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\t{time_stamp1}\n")
            except BaseException as ex:
                print(f"darkcount() failed: {ex}")
                data.write(f"darkcount() failed: {ex}\n")
    await hvgate1off()
    await hvgate2off()
    await pmt1hvdisable()
    await pmt2hvdisable()
    return(0)

async def pmttest3a(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211208: pmttest measurement
	50µs Window, with Flash
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs; reseting low range capa
    gatedelay = 25        # 250ns; switching time for input gate
    meastime = 5000       # 50µs; measurement time, extended from 30µs due to delayed flash lamp 
    delaytime = 5000      # 50µs; delay between reset and offset measurement
    integdelay = 2000     # 20µs; fixed range delay
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest3a'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(gatedelay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    #seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 18
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"pmttest3a() failed: {ex}")
        await asyncio.sleep(delay)
    return resultsum

async def pmttest3al(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211208: pmttest measurement
	50µs Window, with Flash, low range only
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs; reseting low range capa
    gatedelay = 25        # 250ns; switching time for input gate
    meastime = 5000       # 50µs; measurement time, extended from 30µs due to delayed flash lamp 
    delaytime = 5000      # 50µs; delay between reset and offset measurement
    integdelay = 2000     # 20µs; fixed range delay
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest3al'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(gatedelay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 18
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"pmttest3al() failed: {ex}")
        await asyncio.sleep(delay)
    return resultsum

async def pmttest3ah(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211208: pmttest measurement
	50µs Window, with Flash, high range only
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs; reseting low range capa
    gatedelay = 25        # 250ns; switching time for input gate
    meastime = 5000       # 50µs; measurement time, extended from 30µs due to delayed flash lamp 
    delaytime = 5000      # 50µs; delay between reset and offset measurement
    integdelay = 2000     # 20µs; fixed range delay
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest3ah'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(gatedelay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 18
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"pmttest3ah() failed: {ex}")
        await asyncio.sleep(delay)
    return resultsum

async def pmttest3b(iterations, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10):
    '''
    ml 211208: pmttest measurement
	50µs Window, no Flash
    10ns timer tick
    1µs = 100 ticks
    1ms = 100000 ticks
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = 0.1
    await pmt1setdl(dl1)
    await pmt2setdl(dl2)
    await pmt1sethv(hv1)
    await pmt2sethv(hv2)
    await pmt1hvenable()
    await pmt2hvenable()
    await hvgate1on()
    await hvgate2on()

    ## IntegratorModes
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111

    lowresetdelay = 5000  # 50µs; reseting low range capa
    gatedelay = 25        # 250ns; switching time for input gate
    meastime = 5000       # 50µs; measurement time, extended from 30µs due to delayed flash lamp 
    delaytime = 5000      # 50µs; delay between reset and offset measurement
    integdelay = 2000     # 20µs; fixed range delay
    #pre_cnt_window = 100    # 1µs
    cnt_window = meastime // pre_cnt_window
    #window_corse, window_fine = divmod(cnt_window, 65536)

    power = 0.8
    # Limit Freqency for High Power Mode
    if frequency > 500 and flash_mode == 1:
        frequency = 500

    # Set the Flash_Pwr
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    # Set ontime based on Power Mode 
    if flash_mode == 1:
        ontime = 120000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1, timeout=1)
    else:
        ontime = 60000
        await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)

    offtime = int(1 / frequency * 1e8 - ontime)

    op_id = 'pmttest3b'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # pmt1, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # pmt1, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # pmt1, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # pmt1, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # pmt1, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # pmt2, count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # pmt2, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # pmt2, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)   # pmt2, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)   # pmt2, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # ref, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # ref, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # ref, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # ref, analog_high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # abs, analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # abs, analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # abs, analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # abs, analog_high_offset
    seq_gen.Loop(flashes)
    seq_gen.TimerWaitAndRestart(ontime)
    #seq_gen.SetSignals(OutputSignal.Flash)
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.TimerWaitAndRestart(lowresetdelay)
    seq_gen.SetIntegratorMode(abs=low_range_reset, ref=low_range_reset, pmt2=low_range_reset, pmt1=low_range_reset)
    seq_gen.TimerWaitAndRestart(delaytime)
    # high range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_high_range, ref=integrate_in_high_range, pmt2=integrate_in_high_range, pmt1=integrate_in_high_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    # channels: pmt1= 0; pmt2= 1; ref= 4; abs= 5
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=9)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=13)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=17)
    # low range offset
    seq_gen.TimerWaitAndRestart(meastime)
    seq_gen.SetIntegratorMode(abs=integrate_in_low_range, ref=integrate_in_low_range, pmt2=integrate_in_low_range, pmt1=integrate_in_low_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=8)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=12)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    #seq_gen.ResetSignals(OutputSignal.Flash)
    # measurement
    seq_gen.TimerWaitAndRestart(gatedelay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    #seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetIntegratorMode(abs=integrate_autorange, ref=integrate_autorange, pmt2=integrate_autorange, pmt1=integrate_autorange)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(cnt_window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.TimerWaitAndRestart(integdelay)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetIntegratorMode(abs=integrate_with_fixed_range, ref=integrate_with_fixed_range, pmt2=integrate_with_fixed_range, pmt1=integrate_with_fixed_range)
    seq_gen.TimerWait()
    # read out analog signals
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=0, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(channel=1, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=7)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=11)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=14)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=15)
    seq_gen.TimerWaitAndRestart(offtime)
    seq_gen.LoopEnd()
    seq_gen.SetIntegratorMode(abs=full_reset, ref=full_reset, pmt2=full_reset, pmt1=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 18)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 18
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
                print(f"pmttest3b() failed: {ex}")
        await asyncio.sleep(delay)
    return resultsum


async def asxscana(start=500, stop=-4200, step=10, iterations=10, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10 ):
    '''
    ml 211027
    rework 211129
    '''
    myname = '.pmt.asxscana'
    meastime = 5000       # 50µs
    cnt_window = meastime // pre_cnt_window

    await asx_home()
    dfp = datapath('ASxScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, myname)
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(start, stop, step):
            pos = i*0.001
            await asx(pos)
            results = await pmttest3a(iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await asx_home()
    print('ASx_Scan Done\n')
    return(0)    

async def asxscanal(start=500, stop=-4200, step=10, iterations=10, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10 ):
    '''
    ml 211027:
    rework 211129
    '''
    myname = '.pmt.asxscanal'
    meastime = 5000       # 50µs
    cnt_window = meastime // pre_cnt_window

    await asx_home()
    dfp = datapath('ASxScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, myname)
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(start, stop, step):
            pos = i*0.001
            await asx(pos)
            results = await pmttest3al(iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await asx_home()
    print('ASx_ScanL Done\n')
    return(0)    

async def asxscanah(start=500, stop=-4200, step=10, iterations=10, flashes=1, flash_mode=1, frequency=100, hv1 = 0.4, hv2 = 0.5, dl1 = 0.25, dl2 = 0.25, pre_cnt_window=10 ):
    '''
    ml 211027:
    rework 211129
    '''
    myname = '.pmt.asxscanah'
    meastime = 5000       # 50µs
    cnt_window = meastime // pre_cnt_window

    await asx_home()
    dfp = datapath('ASxScan')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Start: {1}; Stop: {2}; Step: {3}; Iterations: {4}; Flashes: {5}; Flash-Mode: {6}; Frequency: {7}",filename, start, stop, step, iterations, flashes, flash_mode, frequency)
    report(result, myname)
    result = str.format("HV1: {0}; HV2: {1}; DL1: {2}; DL2: {3}; CntWindow: {4};PreCntWindow: {5}", hv1, hv2, dl1, dl2, cnt_window, pre_cnt_window)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(start, stop, step):
            pos = i*0.001
            await asx(pos)
            results = await pmttest3ah(iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{pos}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    await asx_home()
    print('ASx_ScanH Done\n')
    return(0)    


async def pmteval1():
    '''
    rework 211129
    '''
    myname = '.pmt.pmteval1'
    await st_home()
    report('st_home', myname)
    await asx_close()
    report('asx_close', myname)
    await fisetup(3, 8.2)
    await st_movetowell(4, 4)
    report('st_movetowell(4, 4)', myname)
    await dlscan(dlstart=0, dlstop=255, dlstep=1, window=100, pre_cnt_window=10)
    return(0)

async def pmteval2(dl = 70, row = 4, col = 4, filter_modul = 3, focus_mover = 8.2):
    '''
    rework 211129
    '''
    myname = '.pmt.pmteval2'
    await st_home()
    report('st_home', myname)
    await asx_close()
    report('asx_close', myname)
    await fisetup(filter_modul, focus_mover)
    await st_movetowell(col, row)
    result = str.format("Well {0}, {1}", col, row)
    report(result, myname)
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.0)
    report('asx(1.0)', myname)
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.3)
    report('asx(1.3)', myname)
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.35)
    report('asx(1.35)', myname)
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.4)
    report('asx(1.4)', myname)
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.45)
    report('asx(1.45)', myname)
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx(1.5)
    report('asx(1.5)', myname)
    await hvscan(dl, hvstart=0, hvstop=188, hvstep=1, flashes=100, flash_mode = 0, pre_cnt_window=10)
    await asx_close()
    report('asx_close', myname)
    return(0)

async def pmteval3(dl1 = 0.25, dl2 = 0.25, hv1 = 0.4, hv2 = 0.4, row = 4, col = 4, filter_modul = 6, focus_mover = 8.2):
    '''
    rework 211129
    '''
    myname = '.pmt.pmteval3'
    start=500
    stop=4000
    step=10
    iterations=1
    flashes=100
    flash_mode=0
    frequency=100
    pre_cnt_window=10
    await st_home()
    report('st_home', myname)
    await asx_close()
    report('asx_close', myname)
    await fisetup(filter_modul, focus_mover)
    await st_movetowell(col, row)
    result = str.format("Well {0}, {1}", col, row)
    report(result, myname)
    #await asxscana(start, stop, step, iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window )
    await asxscanal(start, stop, step, iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window )
    await asxscanah(start, stop, step, iterations, flashes, flash_mode, frequency, hv1, hv2, dl1, dl2, pre_cnt_window )
    await asx_close()
    report('asx_close', myname)
    return(0)


async def fibgmeas1(iterations = 100, flashes = 100, dl1 = 0.25, dl2 = 0.25, hv1 = 0.4, hv2 = 0.4):
    '''
    ml 211214:
    measure background for fi-measurements
    '''
    myname = '.alpha.fibgmeas1'
    pre_cnt_window =10
    dfp = datapath('FIBGTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Iterations: {1}; Flashes: {2}; HV1: {3}; HV2: {4}; DL1: {5}; DL2: {6}; High Speed; no Flash",filename, iterations, flashes, hv1, hv2, dl1, dl2)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(iterations):
            results = await pmttest3b(1, flashes, 0, 1000, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    dfp = datapath('FIBGTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Iterations: {1}; Flashes: {2}; HV1: {3}; HV2: {4}; DL1: {5}; DL2: {6}; High Speed; Flash",filename, iterations, flashes, hv1, hv2, dl1, dl2)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(iterations):
            results = await pmttest3a(1, flashes, 0, 1000, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    dfp = datapath('FIBGTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Iterations: {1}; Flashes: {2}; HV1: {3}; HV2: {4}; DL1: {5}; DL2: {6}; High Power; no Flash",filename, iterations, flashes, hv1, hv2, dl1, dl2)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(iterations):
            results = await pmttest3b(1, flashes, 1, 250, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    dfp = datapath('FIBGTest')
    path, filename = os.path.split(dfp)
    result = str.format("Filename: {0}; Iterations: {1}; Flashes: {2}; HV1: {3}; HV2: {4}; DL1: {5}; DL2: {6}; High Power; Flash",filename, iterations, flashes, hv1, hv2, dl1, dl2)
    report(result, myname)
    print (filename)
    with open(dfp, 'a') as data:
        data.write(f"pos\tpmt1_cr\tpmt1_alr\tpmt1_ahr\tpmt1_alo\tpmt1_aho\tpmt2_cr\tpmt2_alr\tpmt2_ahr\tpmt2_alo\tpmt2_aho\tref_alr\tref_ahr\tref_alo\tref_aho\tabs_alr\tabs_ahr\tabs_alo\tabs_aho\n")
        for i in range(iterations):
            results = await pmttest3a(1, flashes, 1, 250, hv1, hv2, dl1, dl2, pre_cnt_window)
            data.write(f"{i}\t{results[0]}\t{results[1]}\t{results[2]}\t{results[3]}\t{results[4]}\t{results[5]}\t{results[6]}\t{results[7]}\t{results[8]}\t{results[9]}\t{results[10]}\t{results[11]}\t{results[12]}\t{results[13]}\t{results[14]}\t{results[15]}\t{results[16]}\t{results[17]}\n")
    return(0)

async def fibgtest1(iterations = 100, dl1 = 0.25, dl2 = 0.25, hv1 = 0.4, hv2 = 0.4):
    '''
    ml 211214:
    measure background for fi-measurements
    '''
    myname = '.alpha.fibgtest1'
    flashes = 100
    pre_cnt_window =10
    fifi = 3
    fifo = 8.2
    delay = 0.1
    await fisetup(fifi, fifo)
    await st_home()
    report('st_home', myname)
    await asx_close()
    report('asx_close', myname)
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await asx_1()
    report('asx_1', '.alpha.fitest1')
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await asx_3()
    report('asx_3', '.alpha.fitest1')
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await st_movetowell(1, 4)
    report('st_movetowell(1, 4)', myname)
    await asx_close()
    report('asx_close', myname)
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await asx_1()
    report('asx_1', '.alpha.fitest1')
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await asx_3()
    report('asx_3', '.alpha.fitest1')
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await st_movetowell(4, 4)
    report('st_movetowell(4, 4)', myname)
    await asx_close()
    report('asx_close', myname)
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await asx_1()
    report('asx_1', '.alpha.fitest1')
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await asx_3()
    report('asx_3', '.alpha.fitest1')
    await fibgmeas1(iterations, flashes, dl1, dl2, hv1, hv2)
    await asx_close()
    report('asx_close', myname)
    return(0)

